%% SPEIS: extract per-sweep impedance blocks, fit early Nyquist to get CPE slope
clear; clc; close all;

% ---------------------------- User settings -----------------------------
dataFile   = 'NaF SPEISNaFC01.mat';   % MAT-file containing a table named SPEISNaFC01
nPerSweep  = 127;                     % number of frequency points per sweep
relErrMax  = 0.03;                    % stop growing fit window when relative error exceeds this
useSmooth  = true;                    % smooth the Re/Im traces before fitting
smoothWin  = 5;                       % moving window length (samples) when smoothing
pzcShift   = 0.05;                    % PZC shift for gold electrode versus Ag/AgCl
verbose    = true;                    % print progress for each sweep
% -----------------------------------------------------------------------

% Load and sanitize
S = load("NaF.mat");
if ~isfield(S,'SPEISNaFC01')
    error('Expected a variable named SPEISNaFC01 in %s.', dataFile);
end
T = S.SPEISNaFC01;

% Remove any zero-frequency rows
if any(strcmpi(T.Properties.VariableNames,'freqHz'))
    T = T(T.freqHz ~= 0, :);
else
    error('Table must contain a variable named freqHz.');
end

% Basic checks
reqVars = {'ReZOhm','ImZOhm','EweV'};
missing = setdiff(reqVars, T.Properties.VariableNames);
if ~isempty(missing)
    error('Missing required variables in table: %s', strjoin(missing, ', '));
end

nRows = height(T);
if mod(nRows, nPerSweep) ~= 0
    error('Row count (%d) is not divisible by nPerSweep (%d).', nRows, nPerSweep);
end
nSweeps = nRows / nPerSweep;

% Reshape into [nPerSweep x 2 x nSweeps]: column 1 = ReZ, column 2 = ImZ
Z = zeros(nPerSweep, 2, nSweeps);
EP = zeros(1, nSweeps);   % mean potential per sweep

% If frequency is ascending per sweep and you want decreasing (Nyquist), flip later
for k = 1:nSweeps
    idx = (k-1)*nPerSweep + (1:nPerSweep);
    re  = T.ReZOhm(idx);
    im  = T.ImZOhm(idx);
    % Flip so the curve goes from high->low frequency if desired
    Z(:,:,k) = flipud([re, im]);
    EP(k)    = mean(T.EweV(idx));
end

% Helper for smoothing (works even without Curve Fitting Toolbox)
applySmooth = @(x) x;
if useSmooth
    if exist('smooth','file') == 2
        applySmooth = @(x) smooth(x);
    else
        % Use simple moving average as a fallback
        applySmooth = @(x) movmean(x, max(1, smoothWin));
    end
end

% Fit early Nyquist segment for each sweep: Im vs Re ~ m*Re + b
cpeSlope = nan(1, nSweeps);
cpeIntercept = nan(1, nSweeps);
cpeRelErr = nan(1, nSweeps);
fitLenUsed = nan(1, nSweeps);

figure; hold on; box on;
title('Nyquist (smoothed) per sweep');
xlabel('Re(Z) [\Omega]'); ylabel('Im(Z) [\Omega]');

for k = 1:nSweeps
    temp = Z(:,:,k);         % [n x 2] : [Re, Im]
    re0  = applySmooth(temp(:,1));
    im0  = applySmooth(temp(:,2));

    plot(re0, im0);  % show trace

    % Grow the fit window until relative error exceeds threshold
    idxEnd = 2;
    bestP = [NaN, NaN]; bestErr = Inf; bestLen = NaN;

    while idxEnd <= nPerSweep
        idx = 1:idxEnd;
        re = re0(idx);
        im = im0(idx);

        p = polyfit(re, im, 1);
        imFit = polyval(p, re);

        relErr = norm(im - imFit) / max(eps, norm(im)); % guard against divide-by-zero

        if verbose
            fprintf('Sweep %02d | 1:%3d -> m = %.4f, b = %.4f, RelErr = %.2f%%\n', ...
                k, idxEnd, p(1), p(2), 100*relErr);
        end

        if relErr < relErrMax
            bestP   = p;
            bestErr = relErr;
            bestLen = idxEnd;
            idxEnd  = idxEnd + 1;     % try to extend
        else
            if verbose
                fprintf('Sweep %02d | RelErr > %.2f%% at %d, stopping growth.\n', ...
                        k, 100*relErrMax, idxEnd);
            end
            break;
        end
    end

    cpeSlope(k)     = bestP(1);
    cpeIntercept(k) = bestP(2);
    cpeRelErr(k)    = bestErr;
    fitLenUsed(k)   = bestLen;
end

% Convert slope to an "experimental CPE exponent" proxy:
% expCPE = 2*atan(|slope|)/pi (as in your original code)
expCPE = 2 * atan(abs(cpeSlope)) / pi;

% Plot CPE exponent vs potential (optionally shifted by PZC)
figure; hold on; box on;
plot(EP(5:12) + pzcShift, expCPE(5:12), 'o-', 'LineWidth', 1.5, 'MarkerSize', 6, ...
    'DisplayName', 'CPE-Experiment');
xlabel('E_{we} (V) + PZC shift');
ylabel('CPE exponent (proxy)');
title('CPE exponent vs potential');
legend('Location', 'best');

% If you previously wanted only a subset (e.g., sweeps 5:12), you can do:
% sel = 5:12;
% plot(EP(sel) + pzcShift, expCPE(sel), 'o-', 'LineWidth', 1.5, 'MarkerSize', 6);

% Optional: summarize results
results = table((1:nSweeps).', EP.', cpeSlope.', cpeIntercept.', cpeRelErr.', fitLenUsed.', ...
    'VariableNames', {'Sweep','EP_V','Slope','Intercept','RelError','FitLength'});
disp(results);
